package com.trendmicro;

import java.net.HttpCookie;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Implements an in memory cookie store ( as the sun implementations stores for the full
 * URI instead of the domain!).
 *
 * @author juergen_kellerer, 2010-05-14
 * @version 1.0
 */
public class ClientCookieStore implements java.net.CookieStore {

	List<URI> uris = new CopyOnWriteArrayList<URI>();
	ConcurrentMap<String, List<HttpCookie>> cookies = new ConcurrentHashMap<String, List<HttpCookie>>();

	String toHost(URI uri) {
		return uri == null ? "" : uri.getHost();
	}

	/**
	 * {@inheritDoc}
	 */
	public void add(URI uri, HttpCookie cookie) {
		List<HttpCookie> list = cookies.get(toHost(uri));
		if (list == null) {
			List<HttpCookie> newList = new CopyOnWriteArrayList<HttpCookie>();
			list = cookies.putIfAbsent(toHost(uri), newList);
			if (list == null) {
				list = newList;
				uris.add(uri);
			}
		}
		list.remove(cookie);
		list.add(cookie);
	}

	/**
	 * {@inheritDoc}
	 */
	public List<HttpCookie> get(URI uri) {
		List<HttpCookie> list = cookies.get(toHost(uri));
		if (list == null)
			list = Collections.emptyList();
		return Collections.unmodifiableList(list);
	}

	/**
	 * {@inheritDoc}
	 */
	public List<HttpCookie> getCookies() {
		ArrayList<HttpCookie> all = new ArrayList<HttpCookie>(25);
		for (List<HttpCookie> list : cookies.values())
			all.addAll(list);
		return Collections.unmodifiableList(all);
	}

	/**
	 * {@inheritDoc}
	 */
	public List<URI> getURIs() {
		return Collections.unmodifiableList(uris);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean remove(URI uri, HttpCookie cookie) {
		String host = toHost(uri);
		List<HttpCookie> list = cookies.get(host);
		if (list != null && list.remove(cookie)) {
			if (list.isEmpty()) {
				cookies.remove(host);
				uris.remove(uri);
			}
			return true;
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean removeAll() {
		uris.clear();
		cookies = new ConcurrentHashMap<String, List<HttpCookie>>();
		return true;
	}
}
