package com.trendmicro;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.ws.handler.LogicalHandler;
import javax.xml.ws.handler.LogicalMessageContext;
import javax.xml.ws.handler.MessageContext;
import java.io.IOException;
import java.net.CookieHandler;
import java.net.CookieManager;
import java.net.CookiePolicy;
import java.net.URI;
import java.util.*;

/**
 * Implements a sharable session handler to be used with JAX-WS Handler chains.
 *
 * @author juergen_kellerer, 2010-05-14
 * @version 1.0
 */
public class ClientSessionHandler implements LogicalHandler<LogicalMessageContext> {

	private static final Logger log = LoggerFactory.getLogger(ClientSessionHandler.class);

	static final String ADDRESS_KEY = "javax.xml.ws.service.endpoint.address";
	static final String COOKIE_REQUEST_HEADER_NAME = "Cookie";
	static final String COOKIE_RESPONSE_HEADER_NAME = "Set-Cookie";

	Map<String, URI> uriCache = new WeakHashMap<String, URI>();
	CookieManager cookieManager = new CookieManager(new ClientCookieStore(), CookiePolicy.ACCEPT_ALL);

	URI getEndpointAddress(MessageContext context) {
		String endpointAddress = (String) context.get(ADDRESS_KEY);

		if (endpointAddress == null) {
			log.warn("Couldn't extract the endpoint address from the context '{}', " +
					"session persistency will fail when invoking the service.", context);
			return null;
		}

		URI uri = uriCache.get(endpointAddress);
		if (uri == null)
			uriCache.put(endpointAddress, uri = URI.create(endpointAddress));

		return uri;
	}

	boolean handleOutboundMessage(LogicalMessageContext context) {
		@SuppressWarnings("unchecked")
		Map<String, List<String>> headers = (Map<String, List<String>>)
				context.get(MessageContext.HTTP_REQUEST_HEADERS);

		if (headers == null)
			headers = new HashMap<String, List<String>>();

		try {
			URI uri = getEndpointAddress(context);
			if (uri != null) {
				headers.putAll(cookieManager.get(uri, headers));

				// Workarround BUG in Sun's JAX-WS client...
				if (headers.get(COOKIE_REQUEST_HEADER_NAME).isEmpty())
					headers.remove(COOKIE_REQUEST_HEADER_NAME);

				context.put(MessageContext.HTTP_REQUEST_HEADERS, headers);
			}
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		return true;
	}

	boolean handleInboundMessage(LogicalMessageContext context) {
		@SuppressWarnings("unchecked")
		Map<String, List<String>> headers = (Map<String, List<String>>)
				context.get(MessageContext.HTTP_RESPONSE_HEADERS);

		if (headers != null) {
			URI uri = getEndpointAddress(context);
			if (uri != null) {
				List<String> cookies = headers.get(COOKIE_RESPONSE_HEADER_NAME);
				if (cookies != null) {
					try {
						cookieManager.put(uri, Collections.singletonMap(
								COOKIE_RESPONSE_HEADER_NAME, cookies));
					} catch (IOException e) {
						throw new RuntimeException(e);
					}
				}
			}
		}

		return true;
	}

	public CookieManager getCookieManager() {
		return cookieManager;
	}

	/**
	 * Registers this client session handler as the system wide default cookie handler.
	 */
	public void registerAsSystemDefaultCookieHandler() {
		CookieHandler.setDefault(cookieManager);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean handleMessage(LogicalMessageContext context) {
		Boolean outbound = (Boolean) context.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);
		if (outbound)
			return handleOutboundMessage(context);
		else
			return handleInboundMessage(context);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean handleFault(LogicalMessageContext context) {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	public void close(MessageContext context) {
	}
}
